﻿using Amx.Core;
using Amx.Logic.Debug;

namespace Files.Smx
{
    using System;
    using System.IO;
    using System.Collections;


    public enum SpFileCompression : byte
    {
        SPFILE_COMPRESSION_NONE = 0,
        SPFILE_COMPRESSION_GZ = 1,
    }

    public class SpFileSections
    {
        private uint nameoffs;	/**< Relative offset into global string table */
        private uint dataoffs;	/**< Offset into the data section of the file */
        private uint size;		/**< Size of the section's entry in the data section */

        public SpFileSections()
        {
            //TODO: implement me
        }

        public void Read(BinaryReader bin)
        {
            if (bin != null)
            {
                this.nameoffs = bin.ReadUInt32();
                this.dataoffs = bin.ReadUInt32();
                this.size = bin.ReadUInt32();
            }
        }

        public uint DataOffset
        {
            get { return dataoffs; }
        }
        public uint Size
        {
            get { return size; }
        }
        public uint NameOffset
        {
            get { return nameoffs; }
        }
    }

    [Flags()]
    public enum SpCodeFlag : int
    {
        SP_FLAG_DEBUG = (1 << 0),
    }

    public enum SpCodeVersion : byte
    {
        SP_CODEVERS_JIT1 = 9,
        SP_CODEVERS_JIT2 = 10,
    }

    internal class SpFileCodeSection
    {
        private uint codesize;		/**< Codesize in bytes */
        private Bits cellsize;		/**< Cellsize in bytes */
        private SpCodeVersion codeversion;	/**< Version of opcodes supported */
        private SpCodeFlag flags;		/**< Flags */
        private uint main;			/**< Address to "main," if any */
        private uint code;			/**< Relative offset to code */

        public SpFileCodeSection()
        {
            //TODO: implement me
        }

        public bool GetFlag(SpCodeFlag flag)
        {
            return ((flags & flag) == flag);
        }

        public void Read(BinaryReader bin)
        {
            if (bin != null)
            {
                this.codesize = bin.ReadUInt32();
                this.cellsize = (Bits)bin.ReadByte();
                this.codeversion = (SpCodeVersion)bin.ReadByte();
                this.flags = (SpCodeFlag)bin.ReadUInt16();
                this.main = bin.ReadUInt32();
                this.code = bin.ReadUInt32();
            }
        }

        public int CodeSize
        {
            get { return (int)codesize; }
        }
        public Bits CellSize
        {
            get { return cellsize; }
        }
        public SpCodeVersion CodeVersion
        {
            get { return codeversion; }
        }
        public SpCodeFlag Flags
        {
            get { return flags; }
        }
        public uint Main
        {
            get { return main; }
        }
        public uint Code
        {
            get { return code; }
        }
    }

    internal class SpFileDataSection
    {
        private uint datasize;		/**< Size of data section in memory */
        private uint memsize;		/**< Total mem required (includes data) */
        private uint data;			/**< File offset to data (helper) */

        public SpFileDataSection()
        {
            //TODO: implement me
        }

        public void Read(BinaryReader bin)
        {
            if (bin != null)
            {
                this.datasize = bin.ReadUInt32();
                this.memsize = bin.ReadUInt32();
                this.data = bin.ReadUInt32();
            }
        }

        /// <summary>
        /// Size of data section in memory
        /// </summary>
        public int DataSize
        {
            get { return (int)datasize; }
        }
        /// <summary>
        /// Total mem required (includes data)
        /// </summary>
        public uint MemSize
        {
            get { return memsize; }
        }
        /// <summary>
        /// File offset to data (helper) 
        /// </summary>
        public uint Data
        {
            get { return data; }
        }
    }

    internal class SpFilePublicsSection
    {
        private uint address;		/**< Address relative to code section */
        private uint name;		/**< Index into nametable */

        public SpFilePublicsSection()
        {
            //TODO: implement me
        }

        public void Read(BinaryReader bin)
        {
            if (bin != null)
            {
                this.address = bin.ReadUInt32();
                this.name = bin.ReadUInt32();
            }
        }

        /// <summary>
        /// Address relative to code section
        /// </summary>
        public uint Address
        {
            get { return address; }
        }
        /// <summary>
        /// Index into nametable
        /// </summary>
        public uint Name
        {
            get { return name; }
        }
    }

    internal class SpFileNativesSection
    {
        private uint name;		/**< Index into nametable */

        public SpFileNativesSection()
        {
            //TODO: implement me
        }

        public void Read(BinaryReader bin)
        {
            if (bin != null)
            {
                this.name = bin.ReadUInt32();
            }
        }
        /// <summary>
        /// Index into nametable
        /// </summary>
        public uint Name
        {
            get { return name; }
        }
    }

    internal class SpFilePubVarsSection
    {
        private uint address;		/**< Address relative to the DAT section */
        private uint name;		/**< Index into nametable */

        public SpFilePubVarsSection()
        {
            //TODO: implement me
        }

        public void Read(BinaryReader bin)
        {
            if (bin != null)
            {
                this.address = bin.ReadUInt32();
                this.name = bin.ReadUInt32();
            }
        }

        /// <summary>
        /// Address relative to the DAT section
        /// </summary>
        public uint Address
        {
            get { return address; }
        }
        /// <summary>
        /// Index into nametable
        /// </summary>
        public uint Name
        {
            get { return name; }
        }
    }

    internal class SpFileTagSection
    {
        private uint tag_id;		/**< Tag ID from compiler */
        private uint name;		/**< Index into nametable */

        public SpFileTagSection()
        {
            //TODO: implement me
        }

        public void Read(BinaryReader bin)
        {
            if (bin != null)
            {
                this.tag_id = bin.ReadUInt32();
                this.name = bin.ReadUInt32();
            }
        }

        /// <summary>
        /// Tag ID from compiler
        /// </summary>
        public int TagId
        {
            get { return (int)tag_id; }
        }
        /// <summary>
        /// Index into nametable
        /// </summary>
        public uint Name
        {
            get { return name; }
        }
    }

    internal class SpFileDebugInfoSection
    {
        private uint num_files;	/**< number of files */
        private uint num_lines;	/**< number of lines */
        private uint num_syms;	/**< number of symbols */
        private uint num_arrays;	/**< number of symbols which are arrays */

        public void Read(BinaryReader bin)
        {
            if (bin != null)
            {
                this.num_files = bin.ReadUInt32();
                this.num_lines = bin.ReadUInt32();
                this.num_syms = bin.ReadUInt32();
                this.num_arrays = bin.ReadUInt32();
            }
        }

        public SpFileDebugInfoSection()
        {
            //TODO: implement me
        }

        /// <summary>
        /// number of files
        /// </summary>
        public int NumberOfFiles
        {
            get { return (int)num_files; }
        }

        /// <summary>
        /// number of lines
        /// </summary>
        public int NumberOfLines
        {
            get { return (int)num_lines; }
        }
        /// <summary>
        /// number of symbols
        /// </summary>
        public int NumberOfSymbols
        {
            get { return (int)num_syms; }
        }
        /// <summary>
        /// number of symbols which are arrays
        /// </summary>
        public int NumberOfArrays
        {
            get { return (int)num_arrays; }
        }
    }

    internal class SpFileDebugFileSection
    {
        private uint addr;		/**< Address into code */
        private uint name;		/**< Offset into debug nametable */

        public SpFileDebugFileSection()
        {
            //TODO: implement me
        }
        public void Read(BinaryReader bin)
        {
            addr = bin.ReadUInt32();
            name = bin.ReadUInt32();
        }

        /// <summary>
        /// Address into code
        /// </summary>
        public uint Address
        {
            get { return addr; }
        }
        /// <summary>
        /// Offset into debug nametable
        /// </summary>
        public uint Name
        {
            get { return name; }
        }
    }

    internal class SpFileDebugLineSection
    {
        private uint addr;		/**< Address into code */
        private uint line;		/**< Line number */

        public SpFileDebugLineSection()
        {
            //TODO: implement me
        }

        public void Read(BinaryReader bin)
        {
            addr = bin.ReadUInt32();
            line = bin.ReadUInt32();
        }

        /// <summary>
        /// Address into code
        /// </summary>
        public uint Address
        {
            get { return addr; }
        }
        /// <summary>
        /// Line number
        /// </summary>
        public uint LineNumber
        {
            get { return line; }
        }
    }

    internal class SpFileDebugSymbolSection
    {
        private int addr;		            /**< Address rel to DAT or stack frame */
        private Int16 tagid;		        /**< Tag id */
        private uint codestart;	            /**< Start scope validity in code */
        private uint codeend;	            /**< End scope validity in code */
        private SymbolFlagType ident;		/**< Variable type */
        private SymbolFlagClass vclass;		/**< Scope class (local vs global) */
        private UInt16 dimcount;	        /**< Dimension count (for arrays) */
        private Hashtable dimensions;    
        private uint name;		            /**< Offset into debug nametable */

        public SpFileDebugSymbolSection()
        {
            //TODO: implement me
        }

        public void Read(BinaryReader bin)
        {
            addr = bin.ReadInt32();
            tagid = bin.ReadInt16();
            codestart = bin.ReadUInt32();
            codeend = bin.ReadUInt32();
            ident = (SymbolFlagType)bin.ReadByte();
            vclass = (SymbolFlagClass)bin.ReadByte();
            dimcount = bin.ReadUInt16();
            name = bin.ReadUInt32();
            dimensions = new Hashtable(dimcount);

            for (int j = 0; j < dimcount; j++)
            {
                var temp = new SpFileDebugArrayDimSection();
                temp.Read(bin);
                dimensions.Add(j,temp);
            }
        }

        public Hashtable Dimensions
        {
            get { return dimensions; }
        }

        /// <summary>
        /// Address rel to DAT or stack frame
        /// </summary>
        public int Address
        {
            get { return addr; }
        }
        /// <summary>
        /// Tag id
        /// </summary>
        public Int16 TagId
        {
            get { return tagid; }
        }
        /// <summary>
        /// Start scope validity in code
        /// </summary>
        public uint CodeStart
        {
            get { return codestart; }
        }
        /// <summary>
        /// End scope validity in code
        /// </summary>
        public uint CodeEnd
        {
            get { return codeend; }
        }
        /// <summary>
        /// Variable type
        /// </summary>
        public SymbolFlagType VarType
        {
            get { return ident; }
        }
        /// <summary>
        /// Scope class (local vs global)
        /// </summary>
        public SymbolFlagClass ScopeClass
        {
            get { return vclass; }
        }
        /// <summary>
        /// Dimension count (for arrays)
        /// </summary>
        public UInt16 DimensionCount
        {
            get { return dimcount; }
        }
        /// <summary>
        /// Offset into debug nametable
        /// </summary>
        public uint Name
        {
            get { return name; }
        }
    }

    internal class SpFileDebugArrayDimSection
    {
        private UInt16 tagId;		/**< Tag id */
        private uint dim;		/**< Size of dimension */

        public SpFileDebugArrayDimSection()
        {
            //TODO: implement me
        }

        public void Read(BinaryReader bin)
        {
            tagId = bin.ReadUInt16();
            dim = bin.ReadUInt32();
        }

        /// <summary>
        /// Tag id
        /// </summary>
        public UInt16 TagId
        {
            get { return tagId; }
        }
        /// <summary>
        /// Size of dimension
        /// </summary>
        public uint Dimensions
        {
            get { return dim; }
        }
    }

}
